<?php

namespace App\Http\Controllers;

use App\Language;
use App\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

class CrudController extends Controller
{
    protected $model;
    protected $route;
    protected $title;

    protected $table_attributes = [];
    protected $attributes = [];
    protected $relations = [];
    protected $filters = [];
    protected $actions = [];

    protected $storeRequest = null;
    protected $updateRequest = null;

    protected $orderBy = 'created_at';

    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $query = $this->model->query();

        if ($this->model instanceof User) {
            $query->where('id', '!=', Auth::id());
        }

        $params = '';
        foreach (request()->all() as $var => $value) {
            if ($var == "query") {
                $query->where(function ($query) use ($value) {
                    foreach ($this->attributes as $attribute => $type) {
                        if ($type == "text" && request()->input('query') != null) {

                            $query->orWhere($attribute, 'like', '%' . $value . '%');
                        }
                    }
                });

            } else if ($var != "page") {
                $query->where($var, $value);
                $params .= '&' . $var . '=' . $value;
            }
        }

        $items = $query->orderByDesc($this->orderBy)->paginate(12);

        if (count($this->table_attributes) == 0)
            $this->table_attributes = $this->attributes;

        $operations = getIndexOperations($this->model);

        $drivers = User::where("role", "driver")->get();

        return view('layouts.index')->with([
            "title" => $this->title,
            "route" => $this->route,
            "attributes" => $this->table_attributes,
            "filters" => $this->filters,
            "actions" => $this->actions,
            "operations" => $operations,
            "items" => $items,
            'params' => $params,
            "drivers" => $drivers
        ]);
    }

    public function create()
    {
        $data = [];
        foreach ($this->relations as $relation => $type) {

            $model = $this->model->$relation()->getRelated();
            $data[$relation] = $model::get();
        }

        $params = [];
        foreach (request()->all() as $var => $value) {
            $params[$var] = $value;
        }
        return view('layouts.create')->with([
            "title" => $this->title,
            "route" => $this->route,
            "attributes" => $this->attributes,
            "relations" => $this->relations,
            "data" => $data,
            "action" => "create",
            "params" => $params,
        ]);
    }

    public function store(Request $request)
    {
        if (!is_null($this->storeRequest)) {
            $this->validate($request, $this->storeRequest->rules(), $this->storeRequest->messages());
        }

        try {
            $item = $this->model;
            $item->fill($request->except($this->model->translatable));

            if (!is_null($this->model->translatable)) {
                foreach ($this->model->translatable as $attr) {

                    $trans = [];
                    foreach (Language::all() as $language) {
                        $var = $attr . "_" . $language->code;
                        $trans[$language->code] = $request->$var;
                    }
                    $item->$attr = $trans;
                }
            }
            if ($request->has("password")) {
                $item->password = bcrypt($request->password);
            }
            $item->save();

            Session::flash('message', trans('app.success message', ['action' => trans('app.create')]));
            Session::flash('class', 'success');

            if ($this->route == "users")
                return redirect()->to(url('admin/'.$this->route, $item->id ). "?role=" . $item->role);
            elseif ($this->route == "options")
                return redirect()->to(url('admin/'.$this->route, $item->id ). "?type=" . $item->type);
            else
                return redirect()->route($this->route . '.show', $item->id);

        } catch
        (Exception $e) {
            dd($e);
            Session::flash('message', trans('app.error message', ['action' => trans('app.create')]));
            Session::flash('class', 'danger');

            return redirect()->back();
        }
    }

    public function show($id)
    {
        $item = $this->model->find($id);

        if ($item) {
            $attributes = $this->attributes;
            $attributes["created_at"] = 'timestamp';

            return view('layouts.show')->with([
                "title" => $this->title,
                "route" => $this->route,
                "attributes" => $attributes,
                "relations" => $this->relations,
                "action" => "show",
                "operations" => ["edit", "delete"],
                "item" => $item,
                "role" => \request('role'),
            ]);
        } else {

            Session::flash('message', trans('app.error message', ['action' => trans('app.show')]));
            Session::flash('class', 'danger');

            return redirect()->route($this->route . '.index');
        }
    }

    public function edit($id)
    {
        $data = [];

        foreach ($this->relations as $relation => $type) {

            $model = $this->model->$relation()->getRelated();
            $data[$relation] = $model::get();
        }

        return view('layouts.edit')->with([
            "title" => $this->title,
            "route" => $this->route,
            "attributes" => $this->attributes,
            "relations" => $this->relations,
            "action" => "edit",
            "item" => $this->model->find($id),
            "data" => $data
        ]);
    }

    public function update(Request $request, $id)
    {
        try {
            if (!is_null($this->updateRequest)) {
                $this->validate($request, $this->updateRequest->rules(), $this->updateRequest->messages());
            }

            $item = $this->model->find($id);
            $item->fill($request->except($this->model->translatable));
            if (!is_null($this->model->translatable)) {
                foreach ($this->model->translatable as $attr) {
                    $trans = [];
                    foreach (Language::all() as $language) {
                        $var = $attr . "_" . $language->code;
                        $trans[$language->code] = $request->$var;
                    }
                    $item->$attr = $trans;
                }
            }

            if ($request->has("password")) {
                $item->password = bcrypt($request->password);
            }
            $item->update();
            Session::flash('message', trans('app.success message', ['action' => trans('app.update')]));
            Session::flash('class', 'success');

            if ($this->route == "users")
                return redirect()->to(url('admin/'.$this->route, $item->id ). "?role=" . $item->role);
            elseif ($this->route == "options")
                return redirect()->to(url('admin/'.$this->route, $item->id ). "?type=" . $item->type);
            else
                return redirect()->route($this->route . '.show', $item->id);


        } catch (\Exception $e) {
            Session::flash('message', trans('app.error message', ['action' => trans('app.update')]));
            Session::flash('class', 'danger');
        }

        return redirect()->route($this->route . '.show', $id);
    }

    public function destroy($id)
    {
        try {
            $data = $this->model->find($id);
            $this->model->find($id)->delete();

            Session::flash('message', trans('app.success message', ['action' => trans('app.delete')]));
            Session::flash('class', 'success');

        } catch (\Exception $e) {

            Session::flash('message', trans('app.error message', ['action' => trans('app.delete')]));
            Session::flash('class', 'danger');
        }

        if ($this->route == "users")
             return redirect()->to(url('admin/'.$this->route ). "?role=" . $data->role);
        if ($this->route == "options")
            return redirect()->to(url('admin/'.$this->route ). "?type=" . $data->type);
        return redirect()->route($this->route . '.index');
    }
}
